﻿using Hims.Api.Models;
using Hims.Api.Utilities;
using Hims.Domain.Services;
using Hims.Shared.DataFilters;
using Hims.Shared.EntityModels;
using Hims.Shared.Library.Enums;
using Microsoft.AspNetCore.Authorization;
using Microsoft.AspNetCore.Mvc;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;

namespace Hims.Api.Controllers
{


    [Authorize]
    [Route("api/ambulance-Receipt")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class AmbulanceReceiptController : BaseController
    {
        /// inheritdoc   


        private readonly IAmbulanceRecieptService ambulanceRecieptService;
        private readonly IAuditLogService auditLogServices;


        public AmbulanceReceiptController(IAmbulanceRecieptService ambulanceRecieptServices, IAuditLogService auditLogService)
        {
            this.ambulanceRecieptService = ambulanceRecieptServices;
            this.auditLogServices = auditLogService;
        }


        [HttpPost]
        [Route("add")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> Add([FromBody] AmbulanceRecieptModel model, [FromHeader] LocationHeader location)
        {
            model = (AmbulanceRecieptModel)EmptyFilter.Handler(model);
            model.LocationId = Convert.ToInt32(location.LocationId);
            var response = await ambulanceRecieptService.InsertAsync(model);
            if (response > 0)
            {

                var auditLogModel = new AuditLogModel
                {
                    AccountId = model.CreatedBy,
                    LogTypeId = (int)LogTypes.AmbulanceReciept,
                    LogFrom = (int)AccountType.Administrator,
                    LogDate = DateTime.Now,
                    LogDescription = $"{model.CreatedByName} has been added Ambulance receipt {model.AmbulanceRecieptId}.",
                    LocationId = model.LocationId
                };
                await this.auditLogServices.LogAsync(auditLogModel);
            }


            return this.Success(response);
        }
        [HttpPost]
        [Route("fetch")]
        [ProducesResponseType(typeof(List<AmbulanceModel>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAsync([FromBody] AmbulanceRecieptModel model, [FromHeader] LocationHeader location)
        {
            model.LocationId = Convert.ToInt32(location.LocationId);
            var ambulance = await this.ambulanceRecieptService.FetchAllAsync(model);
            return ambulance == null ? this.ServerError() : this.Success(ambulance);
        }


        [HttpPost]
        [Route("update")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> UpdateAsync([FromBody] AmbulanceRecieptModel model, [FromHeader] LocationHeader header)
        {
            model = (AmbulanceRecieptModel)EmptyFilter.Handler(model);
            var response = await this.ambulanceRecieptService.UpdateAsync(model);
            if (response > 0)
            {
                var auditLogModel = new AuditLogModel
                {
                    AccountId = model.CreatedBy,
                    LogTypeId = (int)LogTypes.AmbulanceReciept,
                    LogFrom = (int)AccountType.Administrator,
                    LogDate = DateTime.Now,
                    LogDescription = $"{model.CreatedByName} has been updated ambulance Reciept no  {model.RecieptNo}.",
                    LocationId = Convert.ToInt32(header.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLogModel);
            }
            return this.Success(response);
        }
    }

}
